/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef MX__SEGMENT_H
#define MX__SEGMENT_H

#define MX__CACHE_SEGMENTS(req, segments_list, segments_count)		\
do {									\
  if (likely(segments_count == 1)) {					\
    req.segment.segment_ptr = segments_list[0].segment_ptr;		\
    req.segment.segment_length = segments_list[0].segment_length;	\
    req.segments = &req.segment;					\
    req.count = 1;							\
  }									\
  else if (segments_count == 0) {					\
    /* one empty segment is easier to manage */ 			\
    req.segment.segment_length = 0;					\
    req.segments = &req.segment;					\
    req.count = 1;							\
  }									\
  else {								\
    req.segments = mx__clone_segments(segments_list, segments_count);	\
    req.count = segments_count;						\
  };									\
} while (0);
 
static inline
uint32_t
mx__total_segment_length(mx_segment_t *segments, uint32_t count)
{
  if (likely(count == 1))
    return segments[0].segment_length;
  else {
    uint32_t i;
    uint32_t sum;
    for (i = sum = 0; i < count; ++i)
      sum += segments[i].segment_length;
    return sum;
  }
}

static inline
void
mx__copy_from_segments(char *p,
		       mx_segment_t *segments, uint32_t count,
		       uintptr_t memory_context,
		       uint32_t abs_off, uint32_t len)
{
  uint32_t i;
  uint32_t cur_off;
  for (i = 0, cur_off = 0; i < count && len > 0; cur_off += segments[i].segment_length, i++) {
    uint32_t chunk_len;
    if (abs_off >= cur_off + segments[i].segment_length)
      continue;
    chunk_len = segments[i].segment_length - (abs_off - cur_off);
    if (chunk_len > len)
      chunk_len = len;
    mx_memcpy_from_segment(p, segments[i].segment_ptr + abs_off - cur_off, chunk_len,
			   memory_context);
    /* check status */
    p += chunk_len;
    abs_off += chunk_len;
    len -= chunk_len;
  }
}

static inline
void
mx__abs_to_rel(mx_segment_t *segs, uint32_t count, uint32_t abs_off,
	       uint32_t *index, uint32_t *rel_off)
{
  for (*index = 0, *rel_off = 0; *index < count; (*index)++) {
    *rel_off = abs_off;
    if (abs_off < segs[*index].segment_length) {
      return;
    }
    abs_off -= segs[*index].segment_length;
  }
}

static inline
uint32_t
mx__copy_to_segments(mx_segment_t *segs, uint32_t count,
		     uintptr_t memory_context,
		     uint32_t abs_off, char *data, uint32_t len)
{
  uint32_t index;
  uint32_t rel_off;
  mx_segment_t *seg;
  uint32_t l;
  uint32_t accum;

  accum = 0;
  mx__abs_to_rel(segs, count, abs_off, &index, &rel_off);
  seg = &segs[index];
  while (index < count && len > 0) {
    l = (len < seg->segment_length - rel_off)
      ? len : seg->segment_length - rel_off;
    mx_memcpy_to_segment(seg->segment_ptr + rel_off, data, l,
			 memory_context);
    /* check status */
    accum += l;
    if (l == seg->segment_length - rel_off) {
      ++seg;
      ++index;
    }
    len -= l;
    data += l;
    rel_off = 0;
  }
  return accum;
}

static inline
mx_segment_t *
mx__clone_segments(mx_segment_t *segments, uint32_t count)
{
  mx_segment_t *p;

  p = mx_malloc(sizeof (mx_segment_t)*count);
  if (p != NULL) {
    mx_memcpy(p, segments, sizeof (mx_segment_t)*count);
  }
  return p;
}

#ifndef MX_KERNEL
/* self/shmem communication are currently disabled in the kernel,
  * and memory contextes would make to whole thing horrible */
static inline
uint32_t
mx__copy_between_segments(mx_segment_t *recv_segs, uint32_t recv_count,
			  uintptr_t recv_memory_context, uint32_t recv_abs_off,
			  mx_segment_t *send_segs, uint32_t send_count,
			  uintptr_t send_memory_context, uint32_t send_abs_off,
			  uint32_t len)
{
  uint32_t send_index, recv_index;
  uint32_t send_rel_off, recv_rel_off;
  mx_segment_t *send_seg, *recv_seg;
  uint32_t l;
  uint32_t accum = 0;

  mx__abs_to_rel(send_segs, send_count, send_abs_off, &send_index, &send_rel_off);
  send_seg = &send_segs[send_index];
  mx__abs_to_rel(recv_segs, recv_count, recv_abs_off, &recv_index, &recv_rel_off);
  recv_seg = &recv_segs[recv_index];

  while (send_index < send_count && recv_index < recv_count && len > 0) {
    l = (len < send_seg->segment_length - send_rel_off)
      ? len : send_seg->segment_length - send_rel_off;
    l = (l < recv_seg->segment_length - recv_rel_off)
      ? l : recv_seg->segment_length - recv_rel_off;
    mx_memcpy_between_segments((char*)recv_seg->segment_ptr + recv_rel_off,
			       recv_memory_context,
			       (char*)send_seg->segment_ptr + send_rel_off,
			       send_memory_context,
			       l);
    /* check status */
    accum += l;
    send_rel_off += l;
    recv_rel_off += l;
    if (send_rel_off == send_seg->segment_length) {
      ++send_seg;
      ++send_index;
      send_rel_off = 0;
    }
    if (recv_rel_off == recv_seg->segment_length) {
      ++recv_seg;
      ++recv_index;
      recv_rel_off = 0;
    }
    len -= l;
  }
  return accum;
}
#endif /* ~MX_KERNEL */

#endif
